﻿#pragma once
#include "fbxsdk.h"
#include "VertexTypes.h"
#include "FileImporter.h"

namespace RTCam {

class Scene;
class Entity;
class ResourceManager;

// unique_ptr for FBX SDK objects that require Destroy() to be called instead of being deleted
template<typename T>
class FbxPointer : public std::unique_ptr<T, void (*)(T *)>
{
public:
	FbxPointer(T * ptr = nullptr)
		: unique_ptr(ptr, &FbxPointer::Destroy)
	{ }
	FbxPointer(FbxPointer && other)
		: unique_ptr(std::move(other))
	{ }
	FbxPointer & operator=(FbxPointer && other)
	{
		unique_ptr::operator=(std::move(other));
		return *this;
	}
private:
	static void Destroy(T * ptr)
	{
		if (ptr) ptr->Destroy();
	}
};


// Imports fbx scenes
class SceneImporter : public FileImporter
{
public:
	SceneImporter(void);
	~SceneImporter(void);

	void Initialize(const shared_ptr<ResourceManager>& resources);

	// Opens and loads the contents of an fbx file into a scene.
	// Returns null if nothing was loaded or something failed.
	shared_ptr<Scene> SceneImporter::OpenScene(_In_ ID3D11Device1* device);

	// Inits prebuilt test scenes
	shared_ptr<Scene> InitCubeTestScene();
	shared_ptr<Scene> InitLampsTestScene();
	shared_ptr<Scene> InitHierarchyTestScene();
	shared_ptr<Scene> InitDepthTestScene();
	shared_ptr<Scene> InitCubeGridScene();

private:
	shared_ptr<Scene> Import(_In_ ID3D11Device1* device, _In_z_ const char* filename);

	void ProcessNode	(_In_ ID3D11Device1* device, _In_ FbxNode* node, const shared_ptr<Scene>& scene);
	void ProcessAttrib	(_In_ ID3D11Device1* device, _In_ FbxNodeAttribute* attribute, const shared_ptr<Entity>& entity);
	void ProcessMesh	(_In_ ID3D11Device1* device, _In_ FbxMesh* fbxMesh, const shared_ptr<Entity>& entity);
	
	void ProcessVertex(_In_ FbxMesh* fbxMesh, int polygonId, int polygonVertexId, int totalVertexId,
		vector<DirectX::VertexPositionNormalColor>& vertices, vector<uint16_t>& indices);
	
	// Debugging functions
	void PrintTabs(int numTabs);
	FbxString GetAttributeTypeName(FbxNodeAttribute::EType type);
	void PrintAttribute(FbxNodeAttribute* attribute, int numTabs);
	void PrintNode(FbxNode* node, int numTabs);

	FbxPointer<class FbxManager> m_sdkManager;
	unique_ptr<class FbxGeometryConverter> m_geometryConverter;
	
	weak_ptr<ResourceManager> m_resources;
};

}